/*
 *  linux/arch/arm/mach-uniphier/lowpower.c
 *
 *  Copyright (C) 2012 Panasonic Corporation
 *  - Derived from arch/arm/mach-omap2/omap-mpuss-lowpower.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/kernel.h>
#include <linux/io.h>
#include <linux/errno.h>
#include <linux/linkage.h>
#include <linux/smp.h>
#include <linux/suspend.h>
#include <linux/i2c.h>
#include <linux/i2c-dev.h>

#include <asm/cacheflush.h>
#include <asm/tlbflush.h>
#include <asm/smp_scu.h>
#include <asm/system.h>
#include <asm/pgalloc.h>
#include <asm/suspend.h>

#include <mach/backup-ram-layout.h>
#include <mach/powerdomain.h>

extern void __iomem *scu_base_addr(void);
extern void uniphier_cpu_resume(void);
extern int uniphier_finish_suspend(unsigned long);
extern void uniphier_secondary_startup(void);

struct uniphier_cpu_pm_info {
	void __iomem *scu_saved_addr;
	void __iomem *wkup_saved_addr;
};

static DEFINE_PER_CPU(struct uniphier_cpu_pm_info, uniphier_pm_info);
static void __iomem *backup_ram_base;
static void __iomem *lpmctl_base;
static void __iomem *ibmode_base;
#if defined(CONFIG_MACH_MN2WS0251_REF) || defined(CONFIG_MACH_MN2WS0270_REF)
static void __iomem *sb2c0_base;
static void __iomem *bsir1pt_base;

#define	BTS_COM_POFF	0x00056083
#define	BTS_IR_SET		0xFF
#endif

/*
 * Program the wakeup routine address for the CPU0 and CPU1
 * used for OFF or DORMANT wakeup.
 */
static inline void set_cpu_wakeup_addr(unsigned int cpu_id, u32 addr)
{
	struct uniphier_cpu_pm_info *pm_info = &per_cpu(uniphier_pm_info, cpu_id);

	__raw_writel(addr, pm_info->wkup_saved_addr);
}

/*
 * Store the SCU power status value to scratchpad memory
 */
static void scu_pwrst_prepare(unsigned int cpu_id, unsigned int cpu_state)
{
	struct uniphier_cpu_pm_info *pm_info = &per_cpu(uniphier_pm_info, cpu_id);
	u32 scu_pwr_st;

	switch (cpu_state) {
	case PWRDM_POWER_RET:
		scu_pwr_st = SCU_PM_DORMANT;
		break;
	case PWRDM_POWER_OFF:
		scu_pwr_st = SCU_PM_POWEROFF;
		break;
	case PWRDM_POWER_ON:
	case PWRDM_POWER_INACTIVE:
	default:
		scu_pwr_st = SCU_PM_NORMAL;
		break;
	}

	__raw_writel(scu_pwr_st, pm_info->scu_saved_addr);
}

extern void stop_timers(void);
extern void restart_timers(void);
extern void __l2ca_init(void);

extern void lpctl_save_data(void);
extern void lpctl_lowpower_init(void);
extern void lpctl_if_stop(void);

static void prepare_standby_mode(void)
{
#ifdef CONFIG_MACH_MN2WS0220_REF
	int ret;
	struct i2c_adapter *adap = i2c_get_adapter(3);
	unsigned char i2c_buf[2] = {
		(u8)0x60,	//sub address
		(u8)0x03	//write data
	};
	struct i2c_msg	i2c_message = {
		0x28,		//slave addr[6:0]
		0,		//write flag
		2,		//msg length
		i2c_buf		//pointer to msg data
	};

	if( !adap ){
		printk("prepare_standby_mode() err: i2c_get_adapter\n");
	}
	ret = i2c_transfer( adap, &i2c_message, 1);
	if( ret < 0 ){
		printk("prepare_standby_mode() err: i2c_transfer\n");
	}
#elif defined (CONFIG_MACH_MN2WS0251_REF) || defined(CONFIG_MACH_MN2WS0270_REF)
	__raw_writel(BTS_COM_POFF, sb2c0_base);
	__raw_writeb((u8)BTS_IR_SET, bsir1pt_base);
#endif
}

int pwrdm_pre_transition(void)
{
	lpctl_lowpower_init();
	lpctl_save_data();
	lpctl_if_stop();
#ifdef CONFIG_SMP
	while (get_scu_power_mode(scu_base_addr(), 1) != SCU_PM_POWEROFF);
#endif

	stop_timers();
	return 0;
}

int pwrdm_post_transition(void)
{
	__l2ca_init();
	restart_timers();
	return 0;
}

int uniphier_enter_lowpower(unsigned int cpu, unsigned int power_state)
{
	pwrdm_pre_transition();

	set_cpu_wakeup_addr(cpu, virt_to_phys(uniphier_cpu_resume));
	scu_pwrst_prepare(cpu, power_state);

	cpu_suspend(2, uniphier_finish_suspend);

	pwrdm_post_transition();

	return 0;
}

#ifdef CONFIG_SMP
int uniphier_hotplug_cpu(unsigned int cpu, unsigned int power_state)
{
	set_cpu_wakeup_addr(cpu, virt_to_phys(uniphier_secondary_startup));
	scu_pwrst_prepare(cpu, power_state);

	uniphier_finish_suspend(1);

	return 0;
}
#endif

static int lowpower_notifier(struct notifier_block *self, unsigned long cmd, void *v)
{
	switch (cmd) {
	case PM_SUSPEND_PREPARE:
		prepare_standby_mode();
		break;
	default:
		break;
	}

	return NOTIFY_OK;
}

static struct notifier_block lowpower_notifier_block = {
	.notifier_call = lowpower_notifier,
};


void __iomem *uniphier_get_backup_ram_base(void)
{
	return backup_ram_base;
}

void __iomem *uniphier_get_lpmctl_base(void)
{
	return lpmctl_base;
}

void __iomem *uniphier_get_ibmode_base(void)
{
	return ibmode_base;
}


#define IBMODE_PARAM_SIZE_MAX	32

static char ibmode[IBMODE_PARAM_SIZE_MAX];

static int __init ibmode_setup(char *str)
{
	strncpy(ibmode, str, IBMODE_PARAM_SIZE_MAX);
	return 1;
}
__setup("IBMODE=", ibmode_setup);

char *get_ibmode(void)
{
	return ibmode;
}

unsigned int raw_get_ibmode(void)
{
	return (unsigned int)readl(ibmode_base);
}

void reload_ibmode(void) {
	switch (readl(ibmode_base)) {
	case IBMODE_SUSPEND:
		strcpy(ibmode, "suspend");
		break;
	case IBMODE_RESUME:
		strcpy(ibmode, "resume");
		break;
	case IBMODE_NORMAL:
	default:
		strcpy(ibmode, "normal");
		break;
	}
}


#define LPMCTL_OFFSET	0x00009040
#if defined(CONFIG_MACH_MN2WS0251_REF) || defined(CONFIG_MACH_MN2WS0270_REF)
#define SB2C0_OFFSET	0x00000030
#define BSIR1PT_OFFSET	0x00000071
#endif

static int __init uniphier_lowpower_early_init(void)
{
	unsigned int i;
	char * param;
	extern char * envp_init[];

	/* Static mapping, never released */
	backup_ram_base = ioremap(UNIPHIER_SUSPEND_RAM_BASE, 16);
	if (WARN_ON(!backup_ram_base))
		return -ENOMEM;

	lpmctl_base = ioremap(UNIPHIER_SYSTEM_CTL_REG_BASE + LPMCTL_OFFSET, 4);
	if (WARN_ON(!lpmctl_base))
		return -ENOMEM;

#if defined(CONFIG_MACH_MN2WS0251_REF) || defined(CONFIG_MACH_MN2WS0270_REF)
	sb2c0_base = ioremap(UNIPHIER_SG_CTRL_REG_BASE + SB2C0_OFFSET, 4);
	if (WARN_ON(!sb2c0_base))
		return -ENOMEM;

	bsir1pt_base = ioremap(UNIPHIER_SG_CTRL_REG_BASE + BSIR1PT_OFFSET, 1);
	if (WARN_ON(!bsir1pt_base))
		return -ENOMEM;
#endif

	ibmode_base = ioremap(BOOTMODE_BASE, 4);
	if (WARN_ON(!ibmode_base))
		return -ENOMEM;

	if (!ibmode[0]) {
		reload_ibmode();
	}

	param = kmalloc(IBMODE_PARAM_SIZE_MAX, GFP_KERNEL);
	if (!param) {
		printk("set IBMODE: Cannot allocate memory\n");
		return -ENOMEM;
	}
	strcpy(param, "IBMODE=");
	strncpy(param + strlen("IBMODE="), ibmode,
		IBMODE_PARAM_SIZE_MAX - strlen("IBMODE="));

	for (i = 0; envp_init[i]; i++) {
		if (i == CONFIG_INIT_ENV_ARG_LIMIT) {
			printk("Cannot set enviroment 'IBMODE'\n");
			kfree(param);
			return -ENOMEM;
		}
	}
	envp_init[i] = param;

	return 0;
}
early_initcall(uniphier_lowpower_early_init);


int __init uniphier_lowpower_init(void)
{
	struct uniphier_cpu_pm_info *pm_info;

	/* Initilaise per CPU PM information */
	pm_info = &per_cpu(uniphier_pm_info, 0x0);
	pm_info->scu_saved_addr = backup_ram_base + SCU_OFFSET0;
	pm_info->wkup_saved_addr = backup_ram_base + CPU0_WAKEUP_NS_PA_ADDR_OFFSET;

#ifdef CONFIG_SMP
	pm_info = &per_cpu(uniphier_pm_info, 0x1);
	pm_info->scu_saved_addr = backup_ram_base + SCU_OFFSET1;
	pm_info->wkup_saved_addr = backup_ram_base + CPU1_WAKEUP_NS_PA_ADDR_OFFSET;
#endif

	register_pm_notifier(&lowpower_notifier_block);

	return 0;
}
